# 
# Project automation script for resource_optimization_with_multi_links 
# 
# Created for ISE version 12.2
# 
# This file contains several Tcl procedures (procs) that you can use to automate
# your project by running from xtclsh or the Project Navigator Tcl console.
# If you load this file (using the Tcl command: source resource_optimization_with_multi_links.tcl), then you can
# run any of the procs included here.
# 
# This script is generated assuming your project has HDL sources.
# Several of the defined procs won't apply to an EDIF or NGC based project.
# If that is the case, simply remove them from this script.
# 
# You may also edit any of these procs to customize them. See comments in each
# proc for more instructions.
# 
# This file contains the following procedures:
# 
# Top Level procs (meant to be called directly by the user):
#    run_process: you can use this top-level procedure to run any processes
#        that you choose to by adding and removing comments, or by
#        adding new entries.
#    rebuild_project: you can alternatively use this top-level procedure
#        to recreate your entire project, and the run selected processes.
# 
# Lower Level (helper) procs (called under in various cases by the top level procs):
#    show_help: print some basic information describing how this script works
#    add_source_files: adds the listed source files to your project.
#    set_project_props: sets the project properties that were in effect when this
#        script was generated.
#    create_libraries: creates and adds file to VHDL libraries that were defined when
#        this script was generated.
#    set_process_props: set the process properties as they were set for your project
#        when this script was generated.
# 

set myProject "resource_optimization_with_multi_links"
set myScript "resource_optimization_with_multi_links.tcl"

# 
# Main (top-level) routines
# 

# 
# run_process
# This procedure is used to run processes on an existing project. You may comment or
# uncomment lines to control which processes are run. This routine is set up to run
# the Implement Design and Generate Programming File processes by default. This proc
# also sets process properties as specified in the "set_process_props" proc. Only
# those properties which have values different from their current settings in the project
# file will be modified in the project.
# 
proc run_process {} {

   global myScript
   global myProject

   ## put out a 'heartbeat' - so we know something's happening.
   puts "\n$myScript: running ($myProject)...\n"

   if { ! [ open_project ] } {
      return false
   }

   set_process_props
   #
   # Remove the comment characters (#'s) to enable the following commands 
   # process run "Synthesize"
   # process run "Translate"
   # process run "Map"
   # process run "Place & Route"
   #
   puts "Running 'Implement Design'"
   if { ! [ process run "Implement Design" ] } {
      puts "$myScript: Implementation run failed, check run output for details."
      project close
      return
   }
   puts "Running 'Generate Programming File'"
   if { ! [ process run "Generate Programming File" ] } {
      puts "$myScript: Generate Programming File run failed, check run output for details."
      project close
      return
   }

   puts "Run completed."
   project close

}

# 
# rebuild_project
# 
# This procedure renames the project file (if it exists) and recreates the project.
# It then sets project properties and adds project sources as specified by the
# set_project_props and add_source_files support procs. It recreates VHDL Libraries
# as they existed at the time this script was generated.
# 
# It then calls run_process to set process properties and run selected processes.
# 
proc rebuild_project {} {

   global myScript
   global myProject

   project close
   ## put out a 'heartbeat' - so we know something's happening.
   puts "\n$myScript: Rebuilding ($myProject)...\n"

   set proj_exts [ list ise xise gise ]
   foreach ext $proj_exts {
      set proj_name "${myProject}.$ext"
      if { [ file exists $proj_name ] } { 
         file delete $proj_name
      }
   }

   project new $myProject
   set_project_props
   add_source_files
   create_libraries
   puts "$myScript: project rebuild completed."

   run_process

}

# 
# Support Routines
# 

# 
# show_help: print information to help users understand the options available when
#            running this script.
# 
proc show_help {} {

   global myScript

   puts ""
   puts "usage: xtclsh $myScript <options>"
   puts "       or you can run xtclsh and then enter 'source $myScript'."
   puts ""
   puts "options:"
   puts "   run_process       - set properties and run processes."
   puts "   rebuild_project   - rebuild the project from scratch and run processes."
   puts "   set_project_props - set project properties (device, speed, etc.)"
   puts "   add_source_files  - add source files"
   puts "   create_libraries  - create vhdl libraries"
   puts "   set_process_props - set process property values"
   puts "   show_help         - print this message"
   puts ""
}

proc open_project {} {

   global myScript
   global myProject

   if { ! [ file exists ${myProject}.xise ] } { 
      ## project file isn't there, rebuild it.
      puts "Project $myProject not found. Use project_rebuild to recreate it."
      return false
   }

   project open $myProject

   return true

}
# 
# set_project_props
# 
# This procedure sets the project properties as they were set in the project
# at the time this script was generated.
# 
proc set_project_props {} {

   global myScript

   if { ! [ open_project ] } {
      return false
   }

   puts "$myScript: Setting project properties..."

   project set family "Virtex6"
   project set device "xc6vlx240t"
   project set package "ff1156"
   project set speed "-1"
   project set top_level_module_type "HDL"
   project set synthesis_tool "XST (VHDL/Verilog)"
   project set simulator "Modelsim-SE Mixed"
   project set "Preferred Language" "Verilog"
   project set "Enable Message Filtering" "false"

}


# 
# add_source_files
# 
# This procedure add the source files that were known to the project at the
# time this script was generated.
# 
proc add_source_files {} {

   global myScript

   if { ! [ open_project ] } {
      return false
   }

   puts "$myScript: Adding sources to project..."

   xfile add "Constant_Declaration.vhd"
   xfile add "ml605_board.ucf"
   xfile add "sources/bert/Error_Detection_Module.vhd"
   xfile add "sources/bert/Word_Generator.vhd"
   xfile add "sources/control/Reset_Manager.vhd"
   xfile add "sources/serdes/basics/Agnostic_21bits_counter.vhd"
   xfile add "sources/serdes/basics/Counter_21bits.vhd"
   xfile add "sources/serdes/basics/gf16add.vhd"
   xfile add "sources/serdes/basics/gf16mult.vhd"
   xfile add "sources/serdes/deserializer/deinterleave/Reverse_Interleaving.vhd"
   xfile add "sources/serdes/deserializer/demux40_to_120bits/Read_RX_DP_RAM.vhd"
   xfile add "sources/serdes/deserializer/demux40_to_120bits/Xilinx/Virtex6/Demux_40_to_120bits.vhd"
   xfile add "sources/serdes/deserializer/demux40_to_120bits/Xilinx/Virtex6/RX_DP_RAM.ngc"
   xfile add "sources/serdes/deserializer/demux40_to_120bits/Xilinx/Virtex6/RX_DP_RAM.vhd"
   xfile add "sources/serdes/deserializer/descrambler/Descrambler.vhd"
   xfile add "sources/serdes/deserializer/descrambler/Descrambling.vhd"
   xfile add "sources/serdes/deserializer/manual_frame_alignment/Manual_Frame_Alignment.vhd"
   xfile add "sources/serdes/deserializer/manual_frame_alignment/Modulo_40_Counter.vhd"
   xfile add "sources/serdes/deserializer/manual_frame_alignment/Right_Shifter_39b.vhd"
   xfile add "sources/serdes/deserializer/manual_frame_alignment/Write_RX_DP_RAM.vhd"
   xfile add "sources/serdes/deserializer/pattern_search/Pattern_Search.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/ChienSearch.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/Decoding.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/RSDecoder.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/RSTwoErrorsCorrect.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/adder60.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/errorlocpolynomial.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/gf16inverse.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/gf16log.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/gf16shifter.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/lambdadeterminant.vhd"
   xfile add "sources/serdes/deserializer/rsdecoder/syndromes.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDD_x2.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDD_x3.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDD_x4.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDM_x2.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDM_x3.vhd"
   xfile add "sources/serdes/optimization_utilities/Frame_TDM_x4.vhd"
   xfile add "sources/serdes/serializer/interleave/Interleaving.vhd"
   xfile add "sources/serdes/serializer/mux120_to_40bits/RW_TX_DP_RAM.vhd"
   xfile add "sources/serdes/serializer/mux120_to_40bits/Xilinx/Virtex6/Mux_120_to_40bits.vhd"
   xfile add "sources/serdes/serializer/mux120_to_40bits/Xilinx/Virtex6/tx_dp_ram.ngc"
   xfile add "sources/serdes/serializer/mux120_to_40bits/Xilinx/Virtex6/tx_dp_ram.vhd"
   xfile add "sources/serdes/serializer/rs_encoder/Encoding.vhd"
   xfile add "sources/serdes/serializer/rs_encoder/RSEncoder.vhd"
   xfile add "sources/serdes/serializer/rs_encoder/polydivider.vhd"
   xfile add "sources/serdes/serializer/scrambler/Scrambler.vhd"
   xfile add "sources/serdes/serializer/scrambler/Scrambling.vhd"
   xfile add "sources/serdes/serializer/scrambler/Scrambling_Constants.vhd"
   xfile add "sources/serdes/transceiver/xilinx/virtex6/clk_wiz_v1_6.vhd"
   xfile add "sources/serdes/transceiver/xilinx/virtex6/gtx_serdes.vhd"
   xfile add "sources/serdes/transceiver/xilinx/virtex6/mgt_usrclk_source_mmcm.vhd"
   xfile add "sources/serdes/transceiver/xilinx/virtex6/v6_gtxwizard_v1_4.vhd"
   xfile add "sources/serdes/transceiver/xilinx/virtex6/v6_gtxwizard_v1_4_gtx.vhd"
   xfile add "top_multi_links_optimized.vhd"

   # Set the Top Module as well...
   project set top "rtl" "TOP_MULTI_LINKS_OPTIMIZED"

   puts "$myScript: project sources reloaded."

} ; # end add_source_files

# 
# create_libraries
# 
# This procedure defines VHDL libraries and associates files with those libraries.
# It is expected to be used when recreating the project. Any libraries defined
# when this script was generated are recreated by this procedure.
# 
proc create_libraries {} {

   global myScript

   if { ! [ open_project ] } {
      return false
   }

   puts "$myScript: Creating libraries..."


   # must close the project or library definitions aren't saved.
   project save

} ; # end create_libraries

# 
# set_process_props
# 
# This procedure sets properties as requested during script generation (either
# all of the properties, or only those modified from their defaults).
# 
proc set_process_props {} {

   global myScript

   if { ! [ open_project ] } {
      return false
   }

   puts "$myScript: setting process properties..."

   project set "Compiled Library Directory" "\$XILINX/<language>/<simulator>"
   project set "Global Optimization" "Off" -process "Map"
   project set "Use DSP Block" "Auto" -process "Synthesize - XST"
   project set "DCI Update Mode" "Quiet(Off)" -process "Generate Programming File"
   project set "Enable Cyclic Redundancy Checking (CRC)" "true" -process "Generate Programming File"
   project set "Configuration Rate" "2" -process "Generate Programming File"
   project set "Pack I/O Registers/Latches into IOBs" "Off" -process "Map"
   project set "Place And Route Mode" "Route Only" -process "Place & Route"
   project set "Number of Clock Buffers" "32" -process "Synthesize - XST"
   project set "Max Fanout" "100000" -process "Synthesize - XST"
   project set "Use Clock Enable" "Auto" -process "Synthesize - XST"
   project set "Use Synchronous Reset" "Auto" -process "Synthesize - XST"
   project set "Use Synchronous Set" "Auto" -process "Synthesize - XST"
   project set "Enable Hardware Co-Simulation" "false"
   project set "Filter Files From Compile Order" "true"
   project set "Use Custom Project File" "false" -process "Post-Map Check Syntax"
   project set "Use Custom Project File" "false" -process "Post-Place & Route Check Syntax"
   project set "Use Custom Project File" "false" -process "Post-Translate Check Syntax"
   project set "Last Applied Goal" "Balanced"
   project set "Last Applied Strategy" "Xilinx Default (unlocked)"
   project set "Last Unlock Status" "false"
   project set "Manual Compile Order" "false"
   project set "Placer Effort Level" "High" -process "Map"
   project set "Extra Cost Tables" "0" -process "Map"
   project set "LUT Combining" "Off" -process "Map"
   project set "Combinatorial Logic Optimization" "false" -process "Map"
   project set "Starting Placer Cost Table (1-100)" "1" -process "Map"
   project set "Power Reduction" "Off" -process "Map"
   project set "Register Duplication" "Off" -process "Map"
   project set "Property Specification in Project File" "Store all values"
   project set "Reduce Control Sets" "Auto" -process "Synthesize - XST"
   project set "Shift Register Minimum Size" "2" -process "Synthesize - XST"
   project set "Case Implementation Style" "None" -process "Synthesize - XST"
   project set "Mux Extraction" "Yes"
   project set "RAM Extraction" "true" -process "Synthesize - XST"
   project set "ROM Extraction" "true" -process "Synthesize - XST"
   project set "FSM Encoding Algorithm" "Auto" -process "Synthesize - XST"
   project set "Optimization Goal" "Speed" -process "Synthesize - XST"
   project set "Optimization Effort" "Normal" -process "Synthesize - XST"
   project set "Resource Sharing" "true" -process "Synthesize - XST"
   project set "Shift Register Extraction" "true" -process "Synthesize - XST"
   project set "User Browsed Strategy Files" ""
   project set "VHDL Source Analysis Standard" "VHDL-93"
   project set "Working Directory" "."
   project set "JTAG to System Monitor Connection" "Enable" -process "Generate Programming File"
   project set "Other Bitgen Command Line Options" "" -process "Generate Programming File"
   project set "Generate Detailed Package Parasitics" "false" -process "Generate IBIS Model"
   project set "Maximum Signal Name Length" "20" -process "Generate IBIS Model"
   project set "Show All Models" "false" -process "Generate IBIS Model"
   project set "Specify 'define Macro Name and Value" "" -process "Post-Map Check Syntax"
   project set "Specify 'define Macro Name and Value" "" -process "Post-Place & Route Check Syntax"
   project set "Specify 'define Macro Name and Value" "" -process "Post-Translate Check Syntax"
   project set "Specify Search Directories for 'Include" "" -process "Post-Map Check Syntax"
   project set "Specify Search Directories for 'Include" "" -process "Post-Place & Route Check Syntax"
   project set "Specify Search Directories for 'Include" "" -process "Post-Translate Check Syntax"
   project set "Target UCF File Name" "" -process "Back-annotate Pin Locations"
   project set "Ignore User Timing Constraints" "false" -process "Map"
   project set "Use RLOC Constraints" "Yes" -process "Map"
   project set "Other Map Command Line Options" "" -process "Map"
   project set "Use LOC Constraints" "true" -process "Translate"
   project set "Other Ngdbuild Command Line Options" "" -process "Translate"
   project set "Ignore User Timing Constraints" "false" -process "Place & Route"
   project set "Other Place & Route Command Line Options" "" -process "Place & Route"
   project set "BPI Reads Per Page" "1" -process "Generate Programming File"
   project set "Configuration Pin Busy" "Pull Up" -process "Generate Programming File"
   project set "Configuration Clk (Configuration Pins)" "Pull Up" -process "Generate Programming File"
   project set "UserID Code (8 Digit Hexadecimal)" "0xFFFFFFFF" -process "Generate Programming File"
   project set "Configuration Pin CS" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin DIn" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin Done" "Pull Up" -process "Generate Programming File"
   project set "Create ASCII Configuration File" "false" -process "Generate Programming File"
   project set "Create Binary Configuration File" "false" -process "Generate Programming File"
   project set "Create Bit File" "true" -process "Generate Programming File"
   project set "Enable BitStream Compression" "false" -process "Generate Programming File"
   project set "Run Design Rules Checker (DRC)" "true" -process "Generate Programming File"
   project set "Create IEEE 1532 Configuration File" "false" -process "Generate Programming File"
   project set "Create ReadBack Data Files" "false" -process "Generate Programming File"
   project set "Configuration Pin HSWAPEN" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin Init" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin M0" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin M1" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin M2" "Pull Up" -process "Generate Programming File"
   project set "Configuration Pin Program" "Pull Up" -process "Generate Programming File"
   project set "Power Down Device if Over Safe Temperature" "false" -process "Generate Programming File"
   project set "Configuration Pin RdWr" "Pull Up" -process "Generate Programming File"
   project set "Starting Address for Fallback Configuration" "0x00000000" -process "Generate Programming File"
   project set "JTAG Pin TCK" "Pull Up" -process "Generate Programming File"
   project set "JTAG Pin TDI" "Pull Up" -process "Generate Programming File"
   project set "JTAG Pin TDO" "Pull Up" -process "Generate Programming File"
   project set "JTAG Pin TMS" "Pull Up" -process "Generate Programming File"
   project set "Unused IOB Pins" "Pull Down" -process "Generate Programming File"
   project set "Watchdog Timer Mode" "Off" -process "Generate Programming File"
   project set "Security" "Enable Readback and Reconfiguration" -process "Generate Programming File"
   project set "Done (Output Events)" "Default (4)" -process "Generate Programming File"
   project set "Drive Done Pin High" "false" -process "Generate Programming File"
   project set "Enable Outputs (Output Events)" "Default (5)" -process "Generate Programming File"
   project set "Wait for DCI Match (Output Events)" "Auto" -process "Generate Programming File"
   project set "Wait for PLL Lock (Output Events)" "No Wait" -process "Generate Programming File"
   project set "Release Write Enable (Output Events)" "Default (6)" -process "Generate Programming File"
   project set "FPGA Start-Up Clock" "CCLK" -process "Generate Programming File"
   project set "Enable Internal Done Pipe" "false" -process "Generate Programming File"
   project set "Allow Logic Optimization Across Hierarchy" "false" -process "Map"
   project set "Maximum Compression" "false" -process "Map"
   project set "Generate Detailed MAP Report" "false" -process "Map"
   project set "Map Slice Logic into Unused Block RAMs" "false" -process "Map"
   project set "Perform Timing-Driven Packing and Placement" "false"
   project set "Trim Unconnected Signals" "true" -process "Map"
   project set "Create I/O Pads from Ports" "false" -process "Translate"
   project set "Macro Search Path" "" -process "Translate"
   project set "Netlist Translation Type" "Timestamp" -process "Translate"
   project set "User Rules File for Netlister Launcher" "" -process "Translate"
   project set "Allow Unexpanded Blocks" "false" -process "Translate"
   project set "Allow Unmatched LOC Constraints" "false" -process "Translate"
   project set "Allow Unmatched Timing Group Constraints" "false" -process "Translate"
   project set "Add I/O Buffers" "true" -process "Synthesize - XST"
   project set "Global Optimization Goal" "AllClockNets" -process "Synthesize - XST"
   project set "Keep Hierarchy" "No" -process "Synthesize - XST"
   project set "Register Balancing" "No" -process "Synthesize - XST"
   project set "Register Duplication" "true" -process "Synthesize - XST"
   project set "Asynchronous To Synchronous" "false" -process "Synthesize - XST"
   project set "Automatic BRAM Packing" "false" -process "Synthesize - XST"
   project set "BRAM Utilization Ratio" "100" -process "Synthesize - XST"
   project set "Bus Delimiter" "<>" -process "Synthesize - XST"
   project set "Case" "Maintain" -process "Synthesize - XST"
   project set "Cores Search Directories" "" -process "Synthesize - XST"
   project set "Cross Clock Analysis" "false" -process "Synthesize - XST"
   project set "DSP Utilization Ratio" "100" -process "Synthesize - XST"
   project set "Equivalent Register Removal" "true" -process "Synthesize - XST"
   project set "FSM Style" "LUT" -process "Synthesize - XST"
   project set "Generate RTL Schematic" "Yes" -process "Synthesize - XST"
   project set "Generics, Parameters" "" -process "Synthesize - XST"
   project set "Hierarchy Separator" "/" -process "Synthesize - XST"
   project set "HDL INI File" "" -process "Synthesize - XST"
   project set "LUT Combining" "Auto" -process "Synthesize - XST"
   project set "Library Search Order" "" -process "Synthesize - XST"
   project set "Netlist Hierarchy" "As Optimized" -process "Synthesize - XST"
   project set "Optimize Instantiated Primitives" "false" -process "Synthesize - XST"
   project set "Pack I/O Registers into IOBs" "Auto" -process "Synthesize - XST"
   project set "Power Reduction" "false" -process "Synthesize - XST"
   project set "Read Cores" "true" -process "Synthesize - XST"
   project set "LUT-FF Pairs Utilization Ratio" "100" -process "Synthesize - XST"
   project set "Use Synthesis Constraints File" "true" -process "Synthesize - XST"
   project set "Verilog Include Directories" "" -process "Synthesize - XST"
   project set "Verilog 2001" "true"
   project set "Verilog Macros" "" -process "Synthesize - XST"
   project set "Work Directory" "C:/Documents and Settings/mjs59/Desktop/gbt_virtex6_master1/resource_optimization_with_multi_links/xst" -process "Synthesize - XST"
   project set "Write Timing Constraints" "false" -process "Synthesize - XST"
   project set "Other XST Command Line Options" "" -process "Synthesize - XST"
   project set "Timing Mode" "Performance Evaluation" -process "Map"
   project set "Generate Asynchronous Delay Report" "false" -process "Place & Route"
   project set "Generate Clock Region Report" "false" -process "Place & Route"
   project set "Generate Post-Place & Route Power Report" "false" -process "Place & Route"
   project set "Generate Post-Place & Route Simulation Model" "false" -process "Place & Route"
   project set "Power Reduction" "false" -process "Place & Route"
   project set "Place & Route Effort Level (Overall)" "High" -process "Place & Route"
   project set "Auto Implementation Compile Order" "true"
   project set "Auto Implementation Top" "true"
   project set "Equivalent Register Removal" "true" -process "Map"
   project set "Placer Extra Effort" "None" -process "Map"
   project set "Power Activity File" "" -process "Map"
   project set "Retiming" "false" -process "Map"
   project set "Synthesis Constraints File" "" -process "Synthesize - XST"
   project set "RAM Style" "Auto" -process "Synthesize - XST"
   project set "Verbose Property Persistence" "true"
   project set "Encrypt Bitstream" "false" -process "Generate Programming File"
   project set "Output File Name" "TOP_MULTI_LINKS_OPTIMIZED" -process "Generate IBIS Model"
   project set "Timing Mode" "Performance Evaluation" -process "Place & Route"
   project set "Cycles for First BPI Page Read" "1" -process "Generate Programming File"
   project set "Enable Debugging of Serial Mode BitStream" "false" -process "Generate Programming File"
   project set "Create Logic Allocation File" "false" -process "Generate Programming File"
   project set "Create Mask File" "false" -process "Generate Programming File"
   project set "Watchdog Timer Value" "0x000000" -process "Generate Programming File"
   project set "Allow SelectMAP Pins to Persist" "false" -process "Generate Programming File"
   project set "Enable Multi-Threading" "Off" -process "Map"
   project set "Move First Flip-Flop Stage" "true" -process "Synthesize - XST"
   project set "Move Last Flip-Flop Stage" "true" -process "Synthesize - XST"
   project set "ROM Style" "Auto" -process "Synthesize - XST"
   project set "Safe Implementation" "No" -process "Synthesize - XST"
   project set "AES Initial Vector" "" -process "Generate Programming File"
   project set "Power Activity File" "" -process "Place & Route"
   project set "Extra Effort (Highest PAR level only)" "None" -process "Place & Route"
   project set "HMAC Key (Hex String)" "" -process "Generate Programming File"
   project set "Encrypt Key Select" "BBRAM" -process "Generate Programming File"
   project set "AES Key (Hex String)" "" -process "Generate Programming File"
   project set "Input Encryption Key File" "" -process "Generate Programming File"
   project set "Fallback Reconfiguration" "Enable" -process "Generate Programming File"
   project set "Enable Multi-Threading" "Off" -process "Place & Route"

   puts "$myScript: project property values set."

} ; # end set_process_props

proc main {} {

   if { [llength $::argv] == 0 } {
      show_help
      return true
   }

   foreach option $::argv {
      switch $option {
         "show_help"           { show_help }
         "run_process"         { run_process }
         "rebuild_project"     { rebuild_project }
         "set_project_props"   { set_project_props }
         "add_source_files"    { add_source_files }
         "create_libraries"    { create_libraries }
         "set_process_props"   { set_process_props }
         default               { puts "unrecognized option: $option"; show_help }
      }
   }
}

if { $tcl_interactive } {
   show_help
} else {
   if {[catch {main} result]} {
      puts "$myScript failed: $result."
   }
}

